﻿/*  Author:     Ben Hall
 *  File:       MainPage.Combat.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    MainPage.Combat.cs is a partial class that contains the code needed for handling combat
 *              in the program.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Media.Imaging;
using System.IO;

namespace OnlineGame
{
    public partial class MainPage
    {
        public static AICharacter[] compTeam;
        public static GameCharacter[] turnOrder;
        static GameCharacter selectedComp;
        static int selectedCompPosition;

        public static String playerCommand;
        static int currentTurn;
        static bool playerAlive;
        static bool ranAway;
        static bool turnOrderChanged;

        /*  PopulateItemComboBox() adds all usable items the player has to the ItemComboBox
         */
        private void PopulateItemComboBox()
        {
            for (int i = 0; i < itemList.Length; i++)
            {
                if ((itemList[i] != null) && (playerCharacter.HasItem(i)))
                {
                    //add to combo box
                    String itemToAdd = itemList[i].GetItemName();
                    if (!this.ItemComboBox.Items.Contains(itemToAdd))
                    {
                        this.ItemComboBox.Items.Add(itemToAdd);
                    }
                }
            }
        }

        /*  RemoveItemFromComboBox() removes an item with the itemIndex passed in, from the
         *  ItemComboBox.
         */
        private void RemoveItemFromComboBox(int itemNum)
        {
            String itemToRemove = itemList[itemNum].GetItemName();
            this.ItemComboBox.Items.Remove(itemToRemove);
        }

        /*  RemoveItemFromComboBox() removes an item with the name matching the string passed in,
         *  from the ItemComboBox.
         */
        private void RemoveItemFromComboBox(String itemName)
        {
            this.ItemComboBox.Items.Remove(itemName);
        }

        /*  PopulateSkillComboBox() adds all skills the player knows to the SkillComboBox.
         */
        private void PopulateSkillComboBox()
        {
            for (int i = 0; i < skillList.Length; i++)
            {
                if ((skillList[i] != null) && (playerCharacter.IsSkillKnown(i)))
                {
                    String skillToAdd = skillList[i].GetSkillName();
                    if (!this.SkillComboBox.Items.Contains(skillToAdd))
                    {
                        this.SkillComboBox.Items.Add(skillToAdd);
                    }
                }
            }
        }

        /*  StartCombat() initializes combat and starts a combat thread using the list of AICharacters
         *  passed in.
         */
        public void StartCombat(AICharacter[] encounterIn)
        {
            //load opponents
            compTeam = new AICharacter[0];
            for (int i = 0; i < encounterIn.Length; i++)
            {
                AddToCompTeam(encounterIn[i]);
            }

            currentTurn = 0;
            playerCommand = null;
            DetermineTurnOrder();
            playerAlive = true;
            turnOrderChanged = false;
            ranAway = false;
            AddMessageToChatBox(playerCharacter.GetName() + " has entered combat!", combatMessageColor);
            //set a default target if player goes first
            if (selectedComp == null)
            {
                selectedCompPosition = 0;
                selectedComp = compTeam[selectedCompPosition];
            }
            PopulateSkillComboBox();
            PopulateItemComboBox();
            UpdateCombatDisplay();

            //Create a thread to run the CombatLoop()
            ThreadStart myThreadDelegate = new ThreadStart(CombatLoop);
            Thread myCombatThread = new Thread(myThreadDelegate);
            myCombatThread.Start();
        }

        /*  GetPlayerCommand() returns the current playerCommand, selected by the command
         *  buttons in the CombatGrid.
         */
        public String GetPlayerCommand()
        {
            return playerCommand;
        }

        /*  ResetPlayerCommand() sets playerCommand to null.
         */
        public void ResetPlayerCommand()
        {
            playerCommand = null;
        }

        /*  UpdateCombatDisplay() updates the elements relevant to combat, such as the display
         *  of both player and opponent sides, and CharacterGrid elements.
         */
        private void UpdateCombatDisplay()
        {
            UpdateCharacterGrid();
            ResetCompSide();
            DisplayCompSide();
            DisplayPlayerSide();
        }

        /*  ResetCompSide() resets the visibility of the unselected opponent visual elements
         *  to Collapsed, as well as setting all the borders used in showing current turn,
         *  to a thickness of 0.
         */
        private void ResetCompSide()
        {
            MainCompCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);

            SubCompACombatNameLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompACombatPortrait.Visibility = System.Windows.Visibility.Collapsed;
            SubCompAStatusLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompACombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);

            SubCompBCombatNameLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompBCombatPortrait.Visibility = System.Windows.Visibility.Collapsed;
            SubCompBStatusLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompBCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);

            SubCompCCombatNameLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompCCombatPortrait.Visibility = System.Windows.Visibility.Collapsed;
            SubCompCStatusLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompCCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);

            SubCompDCombatNameLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompDCombatPortrait.Visibility = System.Windows.Visibility.Collapsed;
            SubCompDStatusLabel.Visibility = System.Windows.Visibility.Collapsed;
            SubCompDCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);
        }

        /*  DisplayCompSide() displays the portraits, health and magic values, as well as status
         *  effect information of the AICharacters in combat.  The information of unselected opponents
         *  is in a shortened form.
         *  Opponents that are not alive will also have their portraits dimmed.
         *  It also will select a living opponent (if one exists) if the currently selected opponent
         *  is not alive.
         */
        private void DisplayCompSide()
        {
            //selecting a living opponent, to prevent bug with borders
            if (selectedComp.GetCurrentHealth() <= 0)
            {
                for (int i = 0; i < compTeam.Length; i++)
                {
                    if (compTeam[i].GetCurrentHealth() > 0)
                    {
                        selectedComp = compTeam[i];
                        selectedCompPosition = i;
                        break;
                    }
                }
            }

            MainCompCombatNameLabel.Text = selectedComp.GetName().ToString();
            MainCompCombatPortrait.Source = selectedComp.GetPortrait();
            MainCompCombatHealthValueLabel.Text = selectedComp.GetCurrentHealth().ToString() + " / " + selectedComp.GetMaxHealth().ToString();
            MainCompCombatMagicValueLabel.Text = selectedComp.GetCurrentMagic().ToString() + " / " + selectedComp.GetMaxMagic().ToString();
            MainCompCombatStatusLabel.Text = DisplayMainStatus(selectedComp).ToString();
            if (selectedComp.Equals(turnOrder[currentTurn]))
            {
                MainCompCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(5);
            }
            else
            {
                MainCompCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);
            }
            if (compTeam.Length > 1)
            {
                //display 1-4 other opponents in smaller boxes
                //used to figure out which comp player to place in sub-boxes
                int positionInCompTeam = 0;
                if (compTeam[0].Equals(selectedComp))
                {
                    positionInCompTeam++;
                }
                SubCompACombatNameLabel.Visibility = System.Windows.Visibility.Visible;
                SubCompACombatNameLabel.Text = compTeam[positionInCompTeam].GetName().ToString();

                SubCompACombatPortrait.Visibility = System.Windows.Visibility.Visible;
                SubCompACombatPortrait.Source = compTeam[positionInCompTeam].GetPortrait();
                if (compTeam[positionInCompTeam].GetCurrentHealth() <= 0)
                {
                    SubCompACombatPortrait.Opacity = 0.5;
                }
                else
                {
                    SubCompACombatPortrait.Opacity = 1.0;
                }
                
                SubCompAStatusLabel.Visibility = System.Windows.Visibility.Visible;
                SubCompAStatusLabel.Text = "Health: " + (int)(compTeam[positionInCompTeam].GetHealthPercent() * 100) + "%\n" + "Magic: " + (int)(compTeam[positionInCompTeam].GetMagicPercent() * 100) + "%\n" + DisplaySubStatus(compTeam[positionInCompTeam]);
                if (compTeam[positionInCompTeam].Equals(turnOrder[currentTurn]) && compTeam[positionInCompTeam].GetCurrentHealth() > 0)
                {
                    SubCompACombatPortraitBorder.BorderThickness = new System.Windows.Thickness(5);
                }
                else
                {
                    SubCompACombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);
                }
                positionInCompTeam++;
                if (compTeam.Length > 2)
                {
                    if (compTeam[1].Equals(selectedComp))
                    {
                        positionInCompTeam++;
                    }
                    SubCompBCombatNameLabel.Visibility = System.Windows.Visibility.Visible;
                    SubCompBCombatNameLabel.Text = compTeam[positionInCompTeam].GetName().ToString();
                    
                    SubCompBCombatPortrait.Visibility = System.Windows.Visibility.Visible;
                    SubCompBCombatPortrait.Source = compTeam[positionInCompTeam].GetPortrait();
                    if (compTeam[positionInCompTeam].GetCurrentHealth() <= 0)
                    {
                        SubCompBCombatPortrait.Opacity = 0.5;
                    }
                    else
                    {
                        SubCompBCombatPortrait.Opacity = 1.0;
                    }

                    SubCompBStatusLabel.Visibility = System.Windows.Visibility.Visible;
                    SubCompBStatusLabel.Text = "Health: " + (int)(compTeam[positionInCompTeam].GetHealthPercent() * 100) + "%\n" + "Magic: " + (int)(compTeam[positionInCompTeam].GetMagicPercent() * 100) + "%\n" + DisplaySubStatus(compTeam[positionInCompTeam]);
                    if (compTeam[positionInCompTeam].Equals(turnOrder[currentTurn]) && compTeam[positionInCompTeam].GetCurrentHealth() > 0)
                    {
                        SubCompBCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(5);
                    }
                    else
                    {
                        SubCompBCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);
                    }
                    positionInCompTeam++;
                    if (compTeam.Length > 3)
                    {
                        if (compTeam[2].Equals(selectedComp))
                        {
                            positionInCompTeam++;
                        }
                        SubCompCCombatNameLabel.Visibility = System.Windows.Visibility.Visible;
                        SubCompCCombatNameLabel.Text = compTeam[positionInCompTeam].GetName().ToString();

                        SubCompCCombatPortrait.Visibility = System.Windows.Visibility.Visible;
                        SubCompCCombatPortrait.Source = compTeam[positionInCompTeam].GetPortrait();
                        if (compTeam[positionInCompTeam].GetCurrentHealth() <= 0)
                        {
                            SubCompCCombatPortrait.Opacity = 0.5;
                        }
                        else
                        {
                            SubCompCCombatPortrait.Opacity = 1.0;
                        }

                        SubCompCStatusLabel.Visibility = System.Windows.Visibility.Visible;
                        SubCompCStatusLabel.Text = "Health: " + (int)(compTeam[positionInCompTeam].GetHealthPercent() * 100) + "%\n" + "Magic: " + (int)(compTeam[positionInCompTeam].GetMagicPercent() * 100) + "%\n" + DisplaySubStatus(compTeam[positionInCompTeam]);
                        if (compTeam[positionInCompTeam].Equals(turnOrder[currentTurn]) && compTeam[positionInCompTeam].GetCurrentHealth() > 0)
                        {
                            SubCompCCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(5);
                        }
                        else
                        {
                            SubCompCCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);
                        }
                        positionInCompTeam++;
                        if (compTeam.Length > 4)
                        {
                            if (compTeam[3].Equals(selectedComp))
                            {
                                positionInCompTeam++;
                            }
                            SubCompDCombatNameLabel.Visibility = System.Windows.Visibility.Visible;
                            SubCompDCombatNameLabel.Text = compTeam[positionInCompTeam].GetName().ToString();

                            SubCompDCombatPortrait.Visibility = System.Windows.Visibility.Visible;
                            SubCompDCombatPortrait.Source = compTeam[positionInCompTeam].GetPortrait();
                            if (compTeam[positionInCompTeam].GetCurrentHealth() <= 0)
                            {
                                SubCompDCombatPortrait.Opacity = 0.5;
                            }
                            else
                            {
                                SubCompDCombatPortrait.Opacity = 1.0;
                            }

                            SubCompDStatusLabel.Visibility = System.Windows.Visibility.Visible;
                            SubCompDStatusLabel.Text = "Health: " + (int)(compTeam[positionInCompTeam].GetHealthPercent() * 100) + "%\n" + "Magic: " + (int)(compTeam[positionInCompTeam].GetMagicPercent() * 100) + "%\n" + DisplaySubStatus(compTeam[positionInCompTeam]);
                            if (compTeam[positionInCompTeam].Equals(turnOrder[currentTurn]) && compTeam[positionInCompTeam].GetCurrentHealth() > 0)
                            {
                                SubCompDCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(5);
                            }
                            else
                            {
                                SubCompDCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);
                            }
                            //Max of 5 opponents in combat
                        }
                    }

                }
            }
        }

        /*  DisplayPlayerSide() displays the portrait, health and magic values, as well as status
         *  effect information of the playerCharacter in combat.
         *  If it is the player's turn, a border will be displayed around its portrait.
         */
        //CHANGE: add party data when implemented
        private void DisplayPlayerSide()
        {
            PlayerCombatNameLabel.Text = playerCharacter.GetName();
            PlayerCombatPortrait.Source = playerCharacter.GetPortrait();
            PlayerCombatHealthValueLabel.Text = playerCharacter.GetCurrentHealth().ToString() + " / " + playerCharacter.GetMaxHealth().ToString();
            PlayerCombatMagicValueLabel.Text = playerCharacter.GetCurrentMagic().ToString() + " / " + playerCharacter.GetMaxMagic().ToString();
            PlayerCombatStatusLabel.Text = DisplayMainStatus(playerCharacter).ToString();
            if(playerCharacter.Equals(turnOrder[currentTurn]))
            {
                PlayerCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(5);
            }
            else
            {
                PlayerCombatPortraitBorder.BorderThickness = new System.Windows.Thickness(0);
            }
        }

        /*  DisplaySubStatus() returns a String containing information about the given GameCharacter's
         *  current status effects in a shortened form, for use in unselected opponents or party
         *  members.
         */
        private String DisplaySubStatus(GameCharacter characterIn)
        {
            String status = "";
            if (characterIn.IsCovered())
            {
                status += "C ";
            }
            if (characterIn.IsDefending())
            {
                status += "D ";
            }
            if (characterIn.GetMDefenseChange() != 0)
            {
                if (characterIn.GetMDefenseChange() > 0)
                {
                    status += "MD+ ";
                }
                else
                {
                    status += "MD- ";
                }
            }
            if (characterIn.GetPDefenseChange() != 0)
            {
                if (characterIn.GetPDefenseChange() > 0)
                {
                    status += "PD+ ";
                }
                else
                {
                    status += "PD- ";
                }
            }
            if (characterIn.IsPoisoned())
            {
                status += "P ";
            }
            return status;

        }

        /*  DisplayMainStatus() returns a String containing information about the given GameCharacter's
         *  current status effects in a long form, for use with a selected opponent or the player character.
         */
        private String DisplayMainStatus(GameCharacter characterIn)
        {
            String status = "";
            if (characterIn.IsCovered())
            {
                status += "Covered by " + characterIn.GetCoveredBy().GetName() + "\n";
            }
            if (characterIn.IsCovering())
            {
                status += "Covering " + characterIn.GetCharacterCovered().GetName() + "\n";
            }
            if (characterIn.IsDefending())
            {
                status += "Defending \n";
            }
            if (characterIn.GetMDefenseChange() != 0)
            {
                if (characterIn.GetMDefenseChange() > 0)
                {
                    status += "M.Def Up by " + characterIn.GetMDefenseChange() * 100 + "% \n";
                }
                else if (characterIn.GetMDefenseChange() < 0)
                {
                    status += "M.Def Down by " + characterIn.GetMDefenseChange() * -100 + "% \n";
                }
            }
            if (characterIn.GetPDefenseChange() != 0)
            {
                if (characterIn.GetPDefenseChange() > 0)
                {
                    status += "P.Def Up by " + characterIn.GetPDefenseChange() * 100 + "% \n";
                }
                else if (characterIn.GetMDefenseChange() < 0)
                {
                    status += "P.Def Down by " + characterIn.GetPDefenseChange() * -100 + "% \n";
                }
            }
            if (characterIn.IsPoisoned())
            {
                status += "Poisoned \n";
            }

            return status;
        }

        /*  CombatLoop() is the main loop of combat, repeating until either all opponents are defeated
         *  or the player is defeated or runs away from combat.
         *  
         */
        public void CombatLoop()
        {
            //initial update of display
            this.Dispatcher.BeginInvoke((ThreadStart)delegate()
            {
                UpdateCombatDisplay();
            });
            while (ContinueCombat())
            {
                //EVENTS HAPPENING AT BEGINNING OF ROUND

                GameCharacter currentCharacter = turnOrder[currentTurn];
                //select a target if no opponent is selected
                if (!compTeam.Contains(selectedComp))
                {
                    selectedComp = compTeam[0];
                }

                //Defending status resets at the beginning of each turn.
                if (currentCharacter.IsDefending())
                {
                    this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                    {
                        AddMessageToChatBox(currentCharacter.GetName() + " is no longer defending.", combatMessageColor);
                    });
                    currentCharacter.SetDefending(false);
                }
                //currentCharacter.SetDefending(false);
                //find next turn
                if ((currentCharacter == playerCharacter) && (playerCharacter.GetCurrentHealth() > 0))
                {
                    this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                    {
                        AddMessageToChatBox(playerCharacter.GetName() + "'s turn!", combatMessageColor);
                    });
                    PlayerTurn();
                }
                else
                {
                    AICharacter currentComp = (AICharacter)currentCharacter;
                    //skip if computer is not alive
                    if (currentComp.GetCurrentHealth() > 0)
                    {
                        //CompTurn(currentComp);
                        this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                        {
                            CompTurn(currentComp);
                        });
                    }
                }

                //reorder list if needed
                if (turnOrderChanged == true)
                {
                    //reorder turn order list
                    DetermineTurnOrder();
                    turnOrderChanged = false;
                }

                //END OF TURN EFFECTS (ex. poison)
                if (currentCharacter.IsPoisoned())
                {
                    //lose health equal to 1/20 of max value
                    int poisonDamage = (int)((double)currentCharacter.GetMaxHealth() * 0.05);
                    if (poisonDamage <= 0)
                    {
                        poisonDamage = 1;
                    }
                    this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                    {
                        AddMessageToChatBox(currentCharacter.GetName() + " takes " + poisonDamage + " damage from posion.", combatMessageColor);
                    });
                    int characterHealth = currentCharacter.GetCurrentHealth() - poisonDamage;
                    currentCharacter.SetCurrentHealth(characterHealth);

                    if (currentCharacter.GetCurrentHealth() <= 0)
                    {
                        this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                        {
                            AddMessageToChatBox(currentCharacter.GetName() + " has died from poison!", combatMessageColor);
                        });
                        ResolveDeath(currentCharacter);
                    }
                }

                //update display at end of each round
                this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                {
                    UpdateCombatDisplay();
                });
                IncrementTurn();

                Thread.Sleep(1000);
                if (ContinueCombat() == false)
                {
                    continue;
                }
            }

                //At the end of combat:
                
                if (playerAlive == true)
                {
                    if (ranAway)
                    {
                        this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                        {
                            AddMessageToChatBox("You ran away!", systemMessageColor);
                        });
                    }
                    else
                    {
                        //defeated opponents!
                        
                        this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                        {
                            UpdateCharacterGrid();
                            AddMessageToChatBox("You win!", systemMessageColor);
                        });
                        GiveCombatRewards(playerCharacter);
                        //give rewards
                        
                    }
                }
                else if (playerAlive == false)
                {
                    this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                    {
                        AddMessageToChatBox("You lose...", systemMessageColor);
                    });
                    //return persistent status effects to normal
                    ResolveDeath(playerCharacter);
                    //if the character in is the playerCharacter, reset health and magic to maximum values
                    playerCharacter.SetCurrentHealth(playerCharacter.GetMaxHealth());
                    playerCharacter.SetCurrentMagic(playerCharacter.GetMaxMagic());

                    //defeat penalty -- playerCharacter loses half of current experience)
                    playerCharacter.RemoveExperience(playerCharacter.GetCurrentExperience() / 2);

                }

            //things done at end of combat
            //return non-persistent status effects to normal
            playerCharacter.SetDefending(false);
            //unselect current selectedComp
            selectedComp = null;
            compTeam = null;
            //reset runAway
            playerCharacter.SetRanAway(false);
            ranAway = false;
            this.Dispatcher.BeginInvoke((ThreadStart)delegate()
            {
                ResetCompSide();
                UpdateCharacterGrid();
                SwitchMode();
            });
        }

        /*  GiveCombatRewards() takes in a PlayerCharacter and calls methods to give it experience
         *  and gold rewards earned from combat.
         */
        public void GiveCombatRewards(PlayerCharacter playerCharacterIn)
        {
            AwardExperience(playerCharacterIn);
            AwardCurrency(playerCharacterIn);
        }

        /*  AwardCurrency() takes in a PlayerCharacter and adds an amount of gold equal to the total
         *  gold awarded by all opponents, then sends a message to display this amount to the player.
         */
        public void AwardCurrency(PlayerCharacter playerCharacterIn)
        {
            //get currency from each opponent in combat and add it to the player's total
            int totalCurrencyReward = 0;
            for (int i = 0; i < compTeam.Length; i++)
            {
                totalCurrencyReward += compTeam[i].GetCurrencyCarried();
            }
            playerCharacter.AddCurrency(totalCurrencyReward);
            this.Dispatcher.BeginInvoke((ThreadStart)delegate()
            {
                AddMessageToChatBox(playerCharacter.GetName() + " finds " + totalCurrencyReward + " coins!", systemMessageColor);
            });
        }

        /*  AwardExperience() gives experience to the given PlayerCharacter.
         *  The experience given by an opponent is: 100 + ((opponentLevel - playerLevel) * 10)
         *    with a possible range of 0-200 experience per opponent.
         */
        public void AwardExperience(PlayerCharacter playerCharacterIn)
        {
            //100 experience rewarded for defeating one opponent of equal level
            //-10 for each level the player is above the opponent (minimum 0 experience)
            //+10 for each level the player is below the opponent (maximum double experience)
            //CHANGE:  when adding party combat, exp is based off highest level character OR average level
            int experienceReward = 0;
            int experienceAdjustment = 0;
            int totalExperienceReward = 0;

            for (int i = 0; i < compTeam.Length; i++)
            {
                experienceAdjustment = (compTeam[i].GetLevel() - playerCharacter.GetLevel()) * 10;
                experienceReward = 100 + experienceAdjustment;
                if (experienceReward > 200)
                {
                    experienceReward = 200;
                }
                else if (experienceReward < 0)
                {
                    experienceReward = 0;
                }
                totalExperienceReward += experienceReward;
            }
            playerCharacter.AddExperience(totalExperienceReward);
            this.Dispatcher.BeginInvoke((ThreadStart)delegate()
            {
                AddMessageToChatBox(playerCharacter.GetName() + " earns " + totalExperienceReward + " experience points!", systemMessageColor);
            });
            if (playerCharacter.CheckForLevel())
            {
                this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                {
                    AddMessageToChatBox(LevelUp(playerCharacter), systemMessageColor);
                    UpdateCharacterGrid();
                });
            }

        }

        /*  LevelUp() increases the given PlayerCharacter's level when called, adding stats and skills
         *  when applicable, and returns a string detailing these changes.
         */
        private String LevelUp(PlayerCharacter playerCharacterIn)
        {
            playerCharacterIn.IncreaseLevel();

            //used to display results of the level
            String levelResults = playerCharacterIn.GetName() + " is now level " + playerCharacterIn.GetLevel() + "!\n";
            //returns a string in the form: level,strIncrease,stmIncrease,aglIncrease,intIncrease,fthIncrease,prsIncrease,skillLearned
            String levelUpDetails = GetLevelUpDetails(playerCharacterIn.GetLevel());
            //split string by the ',' character
            String[] levelUpParts = levelUpDetails.Split(',');
            int strUp = Convert.ToInt32(levelUpParts[1]);
            if (strUp > 0)
            {
                levelResults += "Strength increases by " + strUp + ".\n";
            }
            int stmUp = Convert.ToInt32(levelUpParts[2]);
            if (stmUp > 0)
            {
                levelResults += "Stamina increases by " + stmUp + ".\n";
            }
            int aglUp = Convert.ToInt32(levelUpParts[3]);
            if (aglUp > 0)
            {
                levelResults += "Agility increases by " + aglUp + ".\n";
            }           
            int intUp = Convert.ToInt32(levelUpParts[4]);
            if (intUp > 0)
            {
                levelResults += "Intelligence increases by " + intUp + ".\n";
            }
            int fthUp = Convert.ToInt32(levelUpParts[5]);
            if (fthUp > 0)
            {
                levelResults += "Faith increases by " + fthUp + ".\n";
            }
            int prsUp = Convert.ToInt32(levelUpParts[6]);
            if (prsUp > 0)
            {
                levelResults += "Persona increases by " + prsUp + ".\n";
            }
            if (!levelUpParts[7].Equals("NONE"))
            {
                playerCharacterIn.AddSkill(levelUpParts[7]);
                levelResults += playerCharacterIn.GetName() + " learns " + levelUpParts[7] + "!";
            }
            playerCharacterIn.LevelUp(strUp, stmUp, aglUp, intUp, fthUp, prsUp);

            return levelResults;


            //send copy to save to server
        }

        /*  GetLevelUpDetails() returns a String giving the details of a specified level in a Template.txt
         *  resource file based on the Character's class
         */
        private string GetLevelUpDetails(int levelIn)
        {
            //change for each character type
            try
            {
                String classFileToLoad = "GameData/";
                if (playerCharacter.GetPlayerClass() == PlayerClass.Fighter)
                {
                    classFileToLoad += "FighterLevelTemplate.txt";
                }
                else if (playerCharacter.GetPlayerClass() == PlayerClass.Rogue)
                {
                    classFileToLoad += "RogueLevelTemplate.txt";
                }
                else if (playerCharacter.GetPlayerClass() == PlayerClass.Cleric)
                {
                    classFileToLoad += "ClericLevelTemplate.txt";
                }
                else if (playerCharacter.GetPlayerClass() == PlayerClass.Wizard)
                {
                    classFileToLoad += "WizardLevelTemplate.txt";
                }
                Uri uriTextFile = new Uri(classFileToLoad, UriKind.RelativeOrAbsolute);
                StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);
                String levelLineLoaded = "";
                for (int i = 0; i < levelIn; i++)
                {
                    levelLineLoaded = sr.ReadLine();
                }
                return levelLineLoaded;
                
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
            return "";
        }

        /*  IncrementTurn() increments currentTurn and sets the value of currentTurn to 0 if the value
         *  is greater than the number of characters in combat.
         */
        private static void IncrementTurn()
        {
            currentTurn++;
            if (currentTurn > (turnOrder.Length - 1))
            {
                currentTurn = 0;
            }
        }

        /*  RedetermineTurnOrder() is used when the turn order is changed in the middle of combat.
         *  This is different from DetermineTurnOrder() in that it keeps track of the current Character's
         *  turn then sets the turnOrder to the value where the current character is in the new order
         *  of turnOrder[] 
         */
        private static void RedetermineTurnOrder()
        {
            GameCharacter currentCharacter = turnOrder[currentTurn];
            DetermineTurnOrder();
            for (int i = 0; i < turnOrder.Length; i++)
            {
                if (turnOrder[i].Equals(currentCharacter))
                {
                    currentTurn = i;
                    break;
                }
            }
        }

        /*  DetermineTurnOrder() determines the order that characters in combat will act in, based on their
         *  Agility values.  It ignores non-living characters, adjusting the size of the turnOrder array
         *  if needed.
         */
        //CHANGE: details need reworked for party code
        private static void DetermineTurnOrder()
        {
            int livingComps = 0;
            for (int i = 0; i < compTeam.Length; i++)
            {
                if (compTeam[i].GetCurrentHealth() > 0)
                {
                    livingComps++;
                }
            }
            int livingPlayers = 0;
            //REPLACE WHEN ADDING PARTY CODE
            if (playerCharacter.GetCurrentHealth() > 0)
            {
                livingPlayers = 1;
            }
            turnOrder = new GameCharacter[livingPlayers + livingComps];
            int currentPositionInTurnOrder = 0;
            //go backwards from the highest allowable value for Agility to the lowest, inserting characters into turnOrder[] where applicable
            for (int i = CharStat.MAX_VALUE; i >= CharStat.MIN_VALUE; i--)
            {
                if ((playerCharacter.GetAgility() == i) && (playerCharacter.GetCurrentHealth() > 0))
                {
                    turnOrder[currentPositionInTurnOrder] = playerCharacter;
                    currentPositionInTurnOrder++;
                    Console.WriteLine("Adding " + playerCharacter.GetName());
                }
                for (int j = 0; j < compTeam.Length; j++)
                {
                    if ((compTeam[j].GetAgility() == i) && (compTeam[j].GetCurrentHealth() > 0))
                    {
                        turnOrder[currentPositionInTurnOrder] = compTeam[j];
                        currentPositionInTurnOrder++;
                        Console.WriteLine("Adding " + compTeam[j].GetName());
                    }
                }
            }
        }

        /*  PlayerTurn() handles the options a player has during their turn and loops the current combat
         *  thread until the player makes a selection.
         */
        private void PlayerTurn()
        {
            ResetPlayerCommand();
            String command = GetPlayerCommand();
            //String command = "";
            while (command == null)
            {
                this.Dispatcher.BeginInvoke((ThreadStart)delegate()
                {
                    command = GetPlayerCommand();
                    if (command == "S")
                    {
                        if (SkillComboBox.SelectedIndex < 0)
                        {
                            //do not accept command if no skill is selected
                            command = null;
                            ResetPlayerCommand();
                        }
                        else
                        {
                            if (!playerCharacter.CanUseSkill(playerCharacter.GetSkillWithName(SkillComboBox.SelectedValue.ToString())))
                            {
                                //if player can't use skill, try selecting again;
                                command = null;
                                ResetPlayerCommand();
                                AddMessageToChatBox("You do not have enough magic to use " + SkillComboBox.SelectedValue.ToString() + ".", systemMessageColor);
                            }
                            
                            //else, call UseSkill()
                        }
                    }
                    if (command == "I")
                    {
                        if (ItemComboBox.SelectedIndex < 0)
                        {
                            //do not accept Item command if no item is selected
                            command = null;
                            ResetPlayerCommand();
                        }

                    }
                });
                Thread.Sleep(100);
                //do nothing until player input
            }
            String turnDoneMessage = "";
            this.Dispatcher.BeginInvoke((ThreadStart)delegate()
            {
                if (command == "F")
                {
                    //Fight command
                    GameCharacter target = selectedComp;
                    if (target != null)
                    {
                        turnDoneMessage = FightCommand(playerCharacter, target);
                    }
                    else
                    {
                        AddMessageToChatBox(selectedComp.GetName(), systemMessageColor);
                        turnDoneMessage = null;
                    }
                }
                else if (command == "S")
                {
                    //Skill command
                    int skillToUse = playerCharacter.GetSkillWithName(SkillComboBox.SelectedValue.ToString());
                    turnDoneMessage = SkillCommand(playerCharacter, skillToUse);
                }
                else if (command == "I")
                {
                    //Item command
                    int itemToUse = playerCharacter.GetItemWithName(ItemComboBox.SelectedValue.ToString());
                    turnDoneMessage = ItemCommand(itemToUse);
                }
                else if (command == "D")
                {
                    //Defend command
                    turnDoneMessage = DefendCommand(playerCharacter);
                }
                else if (command == "R")
                {
                    //Run command
                    ranAway = RunCommand();
                    if (ranAway)
                    {
                        turnDoneMessage = null;
                    }
                    else
                    {
                        turnDoneMessage = "Could not escape!";
                    }
                }

                AddMessageToChatBox(turnDoneMessage, combatMessageColor);
                if (command == "I")
                {
                    //If an item was used, remove one from the number of that item the player has.
                    int itemUsed = playerCharacter.GetItemWithName(ItemComboBox.SelectedValue.ToString());
                    if (playerCharacter.GetItemQuantity(itemUsed) < 1)
                    {
                        //if last item is used, remove item from ComboBox
                        RemoveItemFromComboBox(itemUsed);
                        ItemQuantityLabel.Text = playerCharacter.GetItemQuantity(itemUsed).ToString();
                    }
                    else
                    {
                        //update quantity in ItemQuantityLabel
                        ItemQuantityLabel.Text = playerCharacter.GetItemQuantity(itemUsed).ToString();
                    }
                }
            });

        }

        /*  ContinueCombat() is used to determine if combat should continue to the next turn.
         *  Returns true only when the player is alive and at least one comp is alive.
         */
        //CHANGE: changes needed when adding party combat
        private static Boolean ContinueCombat()
        {
            bool tmp = true;
            //CHANGE WHEN ADDING PARTY CODE
            if (playerCharacter.GetRanAway() == true)
            {
                tmp = false;
            }
            else if (playerCharacter.GetCurrentHealth() <= 0)
            {
                playerAlive = false;
                tmp = false;
            }
            else
            {
                if (LivingCompCount() < 1)
                {
                    tmp = false;
                }
            }
            return tmp;
        }

        /*  LivingCompCount() returns the number of living comp characters remaining in combat.
         */
        public static int LivingCompCount()
        {
            int livingCount = 0;
            for (int i = 0; i < compTeam.Length; i++)
            {
                if (compTeam[i].GetCurrentHealth() > 0)
                {
                    livingCount++;
                }
            }
            return livingCount;
        }
          
        /*  DefendCommand() is used when the Defend command is chosen by a player or comp character
         *  Sets the character's defending value to true and outputs a message.
         */
        public static String DefendCommand(GameCharacter current)
        {
            current.SetDefending(true);
            String test = current.GetName() + " is defending!";
            return test;
        }

        /*  FightCommand() is used when the Fight command is chosen by a player or comp character
         *  Determines the damage dealt by the character using the command, based on the attack power
         *  the character has and the defensive power the defending character has.
         *  Returns a string with the details of the command.
         */
        public static String FightCommand(GameCharacter attacker, GameCharacter defenderIn)
        {
            String messageReturned = "";
            GameCharacter defender = defenderIn;
            if (defenderIn.GetCoveredBy() != null)
            {
                defender = defenderIn.GetCoveredBy();
                messageReturned += defender.GetName() + " steps in front of " + defenderIn.GetName() + " to take the attack!\n";
            }
            int damage = 0;
            int damageRange = (attacker.GetMaxDamage() - attacker.GetMinDamage());
            damage = attacker.GetMinDamage() + rnd.Next(0, damageRange);

            //strength bonus damage formula
            //formula:  Bonus = STR/4
            int strBonusDamage = (attacker.GetStrength() / 4);
            damage += strBonusDamage;

            //PDefense debuff increases damage if < 0
            if (defender.GetPDefenseChange() < 0)
            {
                int PDefenseExtraDamage = (int)(damage * (-defender.GetPDefenseChange()));
                damage += PDefenseExtraDamage;
            }

            //reduce damage by 50% if defending
            if (defender.IsDefending())
            {
                int defendReduction = (int)(damage * 0.5);
                damage -= defendReduction;
            }

            //calculate damage after armor reduction
            int armorReduction = (int)(damage * defender.GetArmor());
            damage -= armorReduction;

            //find defender damage resistance
            //stamina damage reduction formula
            //formula:  Reduction = STM/5
            int stmDamageReduction = (defender.GetStamina() / 5);
            damage -= (stmDamageReduction);

            //reduce damage if PDefenseChange > 0
            if (defender.GetPDefenseChange() > 0)
            {
                int PDefenseReduction = (int)(damage * defender.GetPDefenseChange());
                damage -= (PDefenseReduction);
            }

            //minimum damage of 0 (no healing from attacks)
            if (damage < 0)
            {
                damage = 0;
            }

            messageReturned += attacker.GetName() + " deals " + damage + " damage to " + defender.GetName() + "!\n";
            //damage dealt
            int defenderHealth = defender.GetCurrentHealth() - damage;
            defender.SetCurrentHealth(defenderHealth);
            if (defenderHealth <= 0)
            {
                messageReturned += attacker.GetName() + " has defeated " + defender.GetName() + "!";
                ResolveDeath(defender);
            }

            return messageReturned;
        }

        /*  ResolveDeath() resets variables, such as poisoned or defending status, back to the default
         *  values.  It also redetermines the turn order of the remaining characters in combat.
         */
        public static void ResolveDeath(GameCharacter character)
        {
            //removed temporarily due to bug causing freeze in Combat
            //RedetermineTurnOrder();
            character.SetDefending(false);
            character.SetPoisoned(false);
            character.SetMDefenseChange(0);
            character.SetPDefenseChange(0);
            if ((character.IsCovered()) || (character.IsCovering()))
            {
                RemoveCover(character);
            }
            //set health to 0 if it is negative
            if (character.GetCurrentHealth() < 0)
            {
                character.SetCurrentHealth(0);
            }
        }

        /*  RemoveCover() resets variables related to cover on characters that are either covering or being
         *  othered.
         */
        public static void RemoveCover(GameCharacter character)
        {
            if (character.IsCovered())
            {
                //remove covering status of character covering the one that died
                character.GetCoveredBy().SetCovering(false);
                character.GetCoveredBy().SetCharacterCovered(null);
                character.SetCoveredBy(null);
            }
            if (character.IsCovering())
            {
                character.GetCharacterCovered().SetCoveredBy(null);
                character.SetCharacterCovered(null);
                character.SetCovering(false);
            }
        }

        /*  SkillCommand() is used when the player wants to select a skill to use.  It checks to make sure
         *  the character knows the skill, has enough available magic to use it, and handles selecting a target.
         *  
         *  Returns false when the player cancels a selection, to signal that the player's turn is not yet over.
         */
        public static String SkillCommand(GameCharacter caster, int skillToUse)
        {
            String messageReturned = "";
            Skill selectedSkill = skillList[skillToUse];
            if (selectedSkill.IsBeneficial())
            {
                messageReturned = UseSkill(selectedSkill, caster, caster);
            }
            else
            {
                GameCharacter target = selectedComp;
                if (target != null)
                {
                    messageReturned = UseSkill(selectedSkill, caster, target);
                }
                else
                {
                    messageReturned = "ERROR IN TARGETING";
                    return null;
                }
            }

            return messageReturned;
        }

        /*  UseSkill() handles the resolution of skills when used by a GameCharacter.
         *  It assumes that the caster has the required Magic to use the skill, as well as assuming the target is valid.
         *  All validation must be done before calling UseSkill()
         */
        public static String UseSkill(Skill skillToUse, GameCharacter caster, GameCharacter targetIn)
        {
            String messageReturned = "";
            messageReturned += caster.GetName() + " is using " + skillToUse.GetSkillName() + " on " + targetIn.GetName() + ".";
            GameCharacter target = targetIn;
            int casterMagic = caster.GetCurrentMagic() - skillToUse.GetSkillCost();
            caster.SetCurrentMagic(casterMagic);
            if (skillToUse.GetSkillName().Equals("Cover"))
            {
                messageReturned += "\n" + caster.GetName() + " is using " + skillToUse.GetSkillName() + " on " + targetIn.GetName() + ".";
                caster.SetCovering(true);
                caster.SetCharacterCovered(targetIn);
                targetIn.SetCoveredBy(caster);
            }

            if (skillToUse.isDamaging())
            {
                if ((!skillToUse.IsMagic()) && (targetIn.IsCovered()))
                {
                    target = targetIn.GetCoveredBy();
                    messageReturned += "\n" + target.GetName() + " steps in front of " + targetIn.GetName() + " to take the attack!";
                }
                int damage = 0;
                int baseDamage = skillToUse.GetNumValue();
                //create a random damage range, from 9/10 of base value to 11/10 of base value
                int minDamage = (int)(baseDamage * (9.0 / 10.0));
                int maxDamage = (int)(baseDamage * (11.0 / 10.0));
                int damageRange = maxDamage - minDamage;

                damage += (rnd.Next(damageRange) + minDamage);

                //if skill is magical, factor in caster intelligence, target faith and mDefense changes
                if (skillToUse.IsMagic())
                {
                    //caster intelligence bonus damage
                    //formula:  int / 4
                    int intBonus = (caster.GetIntelligence() / 4);
                    damage += intBonus;

                    //MDefense debuff increases damage if < 0
                    if (target.GetMDefenseChange() < 0)
                    {
                        int MDefenseExtraDamage = (int)(damage * (-target.GetMDefenseChange()));
                        damage += MDefenseExtraDamage;
                    }


                    //target faith resistance
                    //formula: fth / 5
                    int fthDamageReduction = (target.GetFaith() / 5);
                    damage -= fthDamageReduction;

                    //reduce damage if MDefenseChange > 0
                    if (target.GetMDefenseChange() > 0)
                    {
                        int MDefenseReduction = (int)(damage * target.GetMDefenseChange());
                        damage -= (MDefenseReduction);
                    }

                }
                else //if not  magic, factor in caster strength, target vitality and pDefense changes
                {
                    //caster strength bonus damage
                    //formula: str / 4
                    int strBonus = (caster.GetStrength() / 4);
                    damage += strBonus;

                    //PDefense debuff increases damage if < 0
                    if (target.GetPDefenseChange() < 0)
                    {
                        int PDefenseExtraDamage = (int)(damage * (-target.GetPDefenseChange()));
                        Console.WriteLine("  PDefense- adds: " + PDefenseExtraDamage);
                        damage += PDefenseExtraDamage;
                    }


                    //reduce damage by 50% if defending
                    if (target.IsDefending())
                    {
                        int defendReduction = (int)(damage * 0.5);
                        damage -= defendReduction;
                    }

                    //calculate damage after armor reduction
                    int armorReduction = (int)(damage * target.GetArmor());
                    //damage = (int)(damage * (1.0 - defender.getArmor()));
                    damage -= armorReduction;

                    //find defender damage resistance
                    //stamina damage reduction formula
                    //formula:  Reduction = STM/5
                    int stmDamageReduction = (target.GetStamina() / 5);
                    damage -= (stmDamageReduction);

                    //reduce damage if PDefenseChange > 0
                    if (target.GetPDefenseChange() > 0)
                    {
                        int PDefenseReduction = (int)(damage * target.GetPDefenseChange());
                        damage -= (PDefenseReduction);
                    }

                }
                if (damage < 0)
                {
                    damage = 0;
                }

                messageReturned += "\n" + caster.GetName() + " deals " + damage + " points of damage to " + target.GetName() + "!";
                
                int targetHealth = target.GetCurrentHealth() - damage;
                target.SetCurrentHealth(targetHealth);
                if (targetHealth <= 0)
                {
                    messageReturned += "\n" + caster.GetName() + " has defeated " + target.GetName() + "!";
                    ResolveDeath(target);

                }

            }
            else if (skillToUse.IsHealing())
            {
                int heal = 0;
                int baseHeal = skillToUse.GetNumValue();
                //create a random heal range, from 9/10 of base value to 11/10 of base value
                int minHeal = (int)(baseHeal * (9.0 / 10.0));
                int maxHeal = (int)(baseHeal * (11.0 / 10.0));
                int healRange = maxHeal - minHeal;

                heal += (rnd.Next(healRange) + minHeal);
                //if skill is magical, add faith bonus based on caster's skill.
                if (skillToUse.IsMagic())
                {
                    //caster faith bonus heal
                    //formula:  fth / 4
                    int fthBonus = (caster.GetFaith() / 4);
                    heal += fthBonus;
                }
                else //if skill is not magical, add stamina bonus based on caster's skill
                {
                    int stmHealBonusA = (caster.GetStamina() / 4);
                    heal += stmHealBonusA;
                }
                //add bonus healing based on target's stamina
                //formula: stm / 5
                int stmHealBonus = (target.GetStamina() / 5);
                heal += stmHealBonus;

                messageReturned += "\n" + caster.GetName() + " recovers " + heal + " points of health to " + target.GetName() + "!";

                int targetHealth = target.GetCurrentHealth() + heal;
                
                if (targetHealth > target.GetMaxHealth())
                {
                    targetHealth = target.GetMaxHealth();
                }
                target.SetCurrentHealth(targetHealth);

            }
            //status effects and other
            if (skillToUse.GetPoisonEffect() > 0)
            {
                //cause poison with a chance based on the caster's intelligence and target's persona
                double success = ((caster.GetIntelligence() / target.GetPersona()) / 2.0);
                if (rnd.NextDouble() < success)
                {
                    target.SetPoisoned(true);
                    messageReturned += "\n" + target.GetName() + " has been poisoned!";
                }
                else
                {
                    messageReturned += "\n" + target.GetName() + " resisted the poison.";
                }
            }
            else if (skillToUse.GetPoisonEffect() < 0)
            {
                //remove target's poison
                target.SetPoisoned(false);
                messageReturned += "\n" + target.GetName() + "'s poison has been removed.";
            }

            //PHYSICAL DEFENSE CHANGE
            if (skillToUse.GetPDefenseChange() != 0)
            {
                if (skillToUse.GetPDefenseChange() < 0)
                {
                    //success of reducing defense is based on caster's intelligence and target's persona
                    double success = ((caster.GetIntelligence() / target.GetPersona()) / 2.0);
                    if (rnd.NextDouble() < success)
                    {
                        target.SetPDefenseChange(skillToUse.GetPDefenseChange());
                        messageReturned += "\n" + target.GetName() + "'s physical defense has been reduced by " + (skillToUse.GetPDefenseChange() * -100) + "%!";
                    }
                    else
                    {
                        messageReturned += "\n" + "Physical defense reduction failed.";
                    }
                }
                else
                {
                    target.SetPDefenseChange(skillToUse.GetPDefenseChange());
                    messageReturned += "\n" + target.GetName() + "'s physical defense has been increased by " + (skillToUse.GetPDefenseChange() * 100) + "%!";
                }
            }
            //MAGIC DEFENSE CHANGE
            if (skillToUse.GetMDefenseChange() != 0)
            {
                if (skillToUse.GetMDefenseChange() < 0)
                {
                    //success of reducing defense is based on caster's intelligence and target's persona
                    double success = ((caster.GetIntelligence() / target.GetPersona()) / 2.0);
                    if (rnd.NextDouble() < success)
                    {
                        target.SetMDefenseChange(skillToUse.GetMDefenseChange());
                        messageReturned += "\n" + target.GetName() + "'s magical defense has been reduced by " + (skillToUse.GetMDefenseChange() * -100) + "%!";
                    }
                    else  
                    {
                        messageReturned += "\n" + "Magical defense reduction failed.";
                    }
                }
                else            //automatic success when boosting defense
                {
                    target.SetMDefenseChange(skillToUse.GetMDefenseChange());
                    messageReturned += "\n" + target.GetName() + "'s magical defense has been increased by " + (skillToUse.GetMDefenseChange() * 100) + "%!";
                }
            }
            return messageReturned;
        }

        /*  ItemCommand() is used when the player wants to select a skill to use.  It lists the character's
         *  inventory and prompts the user to select an item, calling UseItem() after an item is selected.
         *  When an item is used, this also handles removing one unit of that item from the character's inventory.
         *  
         *  Returns false when the player cancels a selection, to signal that the player's turn is not yet over.
         */
        public static String ItemCommand(int itemToUse)
        {
            String messageReturned = "";
            int numChoice = itemToUse;
            //code to use item
            Item selectedItem = (Item)itemList[numChoice];
            GameCharacter target = null;
            if (selectedItem.IsBeneficial())
            {
                target = playerCharacter;
            }
            else
            {
                target = selectedComp;
                if (target == null)
                {
                    return null;
                }
            }
            messageReturned = UseItem(selectedItem, target);
            playerCharacter.RemoveItem(numChoice);
            return messageReturned;
        }

        /*  UseItem() handles the resolution of items when used by a player.
         *  It assumes that the target is valid, all validation of this must be done before calling UseItem()
         */
        //CHANGE: NEED TO ADD NEW STATUS EFFECTS: PDEF+/-, MDEF+/-
        public static String UseItem(Item itemToUse, GameCharacter targetIn)
        {
            String messageReturned = "";
            messageReturned += playerCharacter.GetName() + " uses " + itemToUse.GetItemName();
            if (targetIn.Equals(playerCharacter))
            {
                messageReturned += ".";
            }
            else
            {
                messageReturned += " on " + targetIn.GetName() + ".";
            }

            GameCharacter target = targetIn;
            if (itemToUse.IsDamaging())
            {
                if ((!itemToUse.IsMagic()) && (targetIn.IsCovered()))
                {
                    target = targetIn.GetCoveredBy();
                    messageReturned += "\n" + target.GetName() + " steps in front of " + targetIn.GetName() + " to take the attack!";
                }

                int damage = itemToUse.GetNumValue();
                //apply damage reduction
                if (itemToUse.IsMagic())
                {
                    //reduction based on faith
                    int fthDamageReduction = (target.GetFaith() / 5);
                    damage -= fthDamageReduction;

                    //reduce damage if MDefenseChange > 0
                    if (target.GetMDefenseChange() > 0)
                    {
                        int MDefenseDamageReduction = (int)(damage * target.GetMDefenseChange());
                        damage -= (MDefenseDamageReduction);
                    }

                }
                else //if not magic
                {
                    //reduce damage by 50% if defending
                    if (target.IsDefending())
                    {
                        int defendReduction = (int)(damage * 0.5);
                        damage -= defendReduction;
                    }

                    //calculate damage after armor reduction
                    int armorReduction = (int)(damage * target.GetArmor());
                    damage -= armorReduction;

                    //find defender damage resistance
                    //stamina damage reduction formula
                    //formula:  Reduction = STM/5
                    int stmDamageReduction = (target.GetStamina() / 5);
                    damage -= (stmDamageReduction);

                    //reduce damage if PDefenseChange > 0
                    if (target.GetPDefenseChange() > 0)
                    {
                        int PDefenseReduction = (int)(damage * target.GetPDefenseChange());
                        damage -= (PDefenseReduction);
                    }
                }

                if (damage < 0)
                {
                    damage = 0;
                }

                messageReturned += "\n" + "The " + itemToUse.GetItemName() + " deals " + damage + " points of damage to " + target.GetName();

                int targetHealth = target.GetCurrentHealth() - damage;
                target.SetCurrentHealth(targetHealth);
                if (targetHealth <= 0)
                {
                    messageReturned += "\n" + playerCharacter.GetName() + " has defeated " + target.GetName() + "!";
                    ResolveDeath(target);
                }
            }
            else if (itemToUse.IsHealing())
            {
                int heal = itemToUse.GetNumValue();
                Console.WriteLine("Base heal of " + heal);
                if (itemToUse.IsMagic())
                {
                    // faith bonus heal
                    //formula:  fth / 5
                    int fthBonus = (playerCharacter.GetFaith() / 5);
                    heal += fthBonus;
                }
                else //if not magical
                {
                    int stmHealBonus = (playerCharacter.GetStamina() / 5);
                    heal += stmHealBonus;
                }

                messageReturned += "\n" + itemToUse.GetItemName() + " restores " + heal + " points of health to " + playerCharacter.GetName() + ".";

                int playerHealth = playerCharacter.GetCurrentHealth() + heal;

                if (playerHealth > playerCharacter.GetMaxHealth())
                {
                    playerHealth = playerCharacter.GetMaxHealth();
                }
                playerCharacter.SetCurrentHealth(playerHealth);

            }
            //status effects
            if (itemToUse.GetPoisonEffect() > 0)
            {
                double success = 1.0 - ((target.GetPersona() / 100) * 0.5);
                if (rnd.NextDouble() < success)
                {
                    target.SetPoisoned(true);
                    messageReturned += "\n" + target.GetName() + " has been poisoned!";
                }
                else
                {
                    messageReturned += "\n" + target.GetName() + " resists the poison.";
                }
            }
            if (itemToUse.GetPoisonEffect() < 0)
            {
                //recover poison
                target.SetPoisoned(false);
                messageReturned += "\n" + target.GetName() + "'s poison has been removed.";
            }

            //PHYSICAL DEFENSE CHANGE
            if (itemToUse.GetPDefenseChange() != 0)
            {
                if (itemToUse.GetPDefenseChange() < 0)
                {
                    //success of reducing defense is based on caster's intelligence and target's persona
                    double success = 1.0 - ((target.GetPersona() / 100) * 0.5);
                    if (rnd.NextDouble() < success)
                    {
                        target.SetPDefenseChange(itemToUse.GetPDefenseChange());
                        messageReturned += "\n" + target.GetName() + "'s physical defense has been reduced by " + (itemToUse.GetPDefenseChange() * -100) + "%!";
                    }
                    else
                    {
                        messageReturned += "\n" + "Physical defense reduction failed.";
                    }
                }
                else
                {
                    target.SetPDefenseChange(itemToUse.GetPDefenseChange());
                    messageReturned += "\n" + target.GetName() + "'s physical defense has been increased by " + (itemToUse.GetPDefenseChange() * 100) + "%!";
                }
            }

            //MAGICAL DEFENSE CHANGE
            if (itemToUse.GetMDefenseChange() != 0)
            {
                if (itemToUse.GetMDefenseChange() < 0)
                {
                    //success of reducing defense is based on caster's intelligence and target's persona
                    double success = 1.0 - ((target.GetPersona() / 100) * 0.5);
                    if (rnd.NextDouble() < success)
                    {
                        target.SetMDefenseChange(itemToUse.GetMDefenseChange());
                        messageReturned += "\n" + target.GetName() + "'s magical defense has been reduced by " + (itemToUse.GetMDefenseChange() * -100) + "%!";
                    }
                    else
                    {
                        messageReturned += "\n" + "Magical defense reduction failed.";
                    }
                }
                else
                {
                    target.SetMDefenseChange(itemToUse.GetMDefenseChange());
                    messageReturned += "\n" + target.GetName() + "'s physical defense has been increased by " + (itemToUse.GetMDefenseChange() * 100) + "%!";
                }
            }

            return messageReturned;
        }

        /*  RunCommand() handles when the player attempts to run from combat.
         *  The chance of success is based on the player's agility compared to the average agility of the 
         *  living comp characters in combat.
         */
        public static bool RunCommand()
        {
            //determine the average agility of the living AICharacters in battle to compare to player agility
            int totalComAgility = 0;
            for (int i = 0; i < compTeam.Length; i++)
            {
                if (compTeam[i].GetCurrentHealth() > 0)
                {
                    totalComAgility += compTeam[i].GetAgility();
                }
            }

            //int averageComAgility = (totalComAgility / compTeam.Length);
            int averageComAgility = (totalComAgility / LivingCompCount());
            double runChance = (((double)playerCharacter.GetAgility() / (double)averageComAgility) / (double)1.5);
            double runRoll = rnd.NextDouble();

            if (runRoll < runChance)
            {
                ranAway = true;
                playerCharacter.SetRanAway(true);
                
                return true;
            }
            else
            {
                Console.WriteLine("Could not run away!");
            }
            return false;
        }

        /*  CompTurn() handles the current comp character's turn, calling its MakeMove() method, which is based
         *  on the type of AI the character is.
         */
        private void CompTurn(AICharacter currentComp)
        {
            String messageToWrite = "";
            AddMessageToChatBox(currentComp.GetName() + "'s turn!", combatMessageColor);
            messageToWrite = currentComp.MakeMove();
            AddMessageToChatBox(messageToWrite, combatMessageColor);
        }

        /*  AddToCompTeam() creates a new AICharacter array one larger than compTeam, transfers over the contents
         *  of compteam to the new array, adds the AICharacter passed in and sets compTeam equal to the new
         *  array
         */
        private static void AddToCompTeam(AICharacter compToAdd)
        {
            //create array one  larger than the current compTeam
            AICharacter[] newTeam = new AICharacter[compTeam.Length + 1];
            //move characters from old team array to new team array
            for (int i = 0; i < compTeam.Length; i++)
            {
                newTeam[i] = compTeam[i];
            }
            //add new compToAdd to the new array
            newTeam[compTeam.Length] = compToAdd;
            //set old array equal to this array
            compTeam = newTeam;
        }

        /*  SetRanAway() sets the ranAway value equal to the boolean passed in
         */
        public void SetRanAway(bool runIn)
        {
            ranAway = runIn;
        }

    }
}
